﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>
#include <aws/iotsitewise/model/AssetModelStatus.h>
#include <aws/iotsitewise/model/AssetModelType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IoTSiteWise {
namespace Model {

/**
 * <p>Contains a summary of an asset model.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iotsitewise-2019-12-02/AssetModelSummary">AWS
 * API Reference</a></p>
 */
class AssetModelSummary {
 public:
  AWS_IOTSITEWISE_API AssetModelSummary() = default;
  AWS_IOTSITEWISE_API AssetModelSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API AssetModelSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the asset model (used with IoT SiteWise API operations).</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  AssetModelSummary& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The external ID of the asset model. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-ids">Using
   * external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetExternalId() const { return m_externalId; }
  inline bool ExternalIdHasBeenSet() const { return m_externalIdHasBeenSet; }
  template <typename ExternalIdT = Aws::String>
  void SetExternalId(ExternalIdT&& value) {
    m_externalIdHasBeenSet = true;
    m_externalId = std::forward<ExternalIdT>(value);
  }
  template <typename ExternalIdT = Aws::String>
  AssetModelSummary& WithExternalId(ExternalIdT&& value) {
    SetExternalId(std::forward<ExternalIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>
   * of the asset model, which has the following format.</p> <p>
   * <code>arn:${Partition}:iotsitewise:${Region}:${Account}:asset-model/${AssetModelId}</code>
   * </p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  AssetModelSummary& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the asset model.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  AssetModelSummary& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of asset model.</p> <ul> <li> <p> <b>ASSET_MODEL</b> – (default) An
   * asset model that you can use to create assets. Can't be included as a component
   * in another asset model.</p> </li> <li> <p> <b>COMPONENT_MODEL</b> – A reusable
   * component that you can include in the composite models of other asset models.
   * You can't create assets directly from this type of asset model. </p> </li> </ul>
   */
  inline AssetModelType GetAssetModelType() const { return m_assetModelType; }
  inline bool AssetModelTypeHasBeenSet() const { return m_assetModelTypeHasBeenSet; }
  inline void SetAssetModelType(AssetModelType value) {
    m_assetModelTypeHasBeenSet = true;
    m_assetModelType = value;
  }
  inline AssetModelSummary& WithAssetModelType(AssetModelType value) {
    SetAssetModelType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The asset model description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  AssetModelSummary& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the asset model was created, in Unix epoch time.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  AssetModelSummary& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the asset model was last updated, in Unix epoch time.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdateDate() const { return m_lastUpdateDate; }
  inline bool LastUpdateDateHasBeenSet() const { return m_lastUpdateDateHasBeenSet; }
  template <typename LastUpdateDateT = Aws::Utils::DateTime>
  void SetLastUpdateDate(LastUpdateDateT&& value) {
    m_lastUpdateDateHasBeenSet = true;
    m_lastUpdateDate = std::forward<LastUpdateDateT>(value);
  }
  template <typename LastUpdateDateT = Aws::Utils::DateTime>
  AssetModelSummary& WithLastUpdateDate(LastUpdateDateT&& value) {
    SetLastUpdateDate(std::forward<LastUpdateDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the asset model.</p>
   */
  inline const AssetModelStatus& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = AssetModelStatus>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = AssetModelStatus>
  AssetModelSummary& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number of the asset model.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  AssetModelSummary& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_externalId;

  Aws::String m_arn;

  Aws::String m_name;

  AssetModelType m_assetModelType{AssetModelType::NOT_SET};

  Aws::String m_description;

  Aws::Utils::DateTime m_creationDate{};

  Aws::Utils::DateTime m_lastUpdateDate{};

  AssetModelStatus m_status;

  Aws::String m_version;
  bool m_idHasBeenSet = false;
  bool m_externalIdHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_assetModelTypeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_creationDateHasBeenSet = false;
  bool m_lastUpdateDateHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
