#!/bin/sh

# Copyright (C) 2024-2025 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/build/kea/src/build/src/share/database/scripts/pgsql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/build/kea/src/build/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check only major version to allow for intermediary backported schema changes.
version=$(pgsql_version "${@}" | cut -d '.' -f 1)
if test "${version}" != '26'; then
    printf 'This script upgrades 26.* to 27.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

psql "$@" >/dev/null <<EOF
START TRANSACTION;

-- This line starts the schema upgrade to version 27.0.

SELECT set_config('kea.disable_audit', 'true', false);

CREATE OR REPLACE FUNCTION textToJSONList(orig TEXT)
    RETURNS TEXT AS \$\$
DECLARE
BEGIN
    IF orig = '' OR orig IS NULL THEN
        RETURN NULL;
    END IF;

    RETURN CONCAT('[ "', orig, '" ]');
END;
\$\$ LANGUAGE plpgsql;

ALTER TABLE dhcp4_shared_network RENAME client_class TO client_classes;
ALTER TABLE dhcp4_shared_network ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp4_shared_network SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp4_subnet RENAME client_class TO client_classes;
ALTER TABLE dhcp4_subnet ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp4_subnet SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp4_pool RENAME client_class TO client_classes;
ALTER TABLE dhcp4_pool ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp4_pool SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp6_shared_network RENAME client_class TO client_classes;
ALTER TABLE dhcp6_shared_network ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp6_shared_network SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp6_subnet RENAME client_class TO client_classes;
ALTER TABLE dhcp6_subnet ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp6_subnet SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp6_pool RENAME client_class TO client_classes;
ALTER TABLE dhcp6_pool ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp6_pool SET client_classes = textToJSONList(client_classes);

ALTER TABLE dhcp6_pd_pool RENAME client_class TO client_classes;
ALTER TABLE dhcp6_pd_pool ALTER COLUMN client_classes TYPE TEXT;

UPDATE dhcp6_pd_pool SET client_classes = textToJSONList(client_classes);

DROP FUNCTION IF EXISTS testToJSONList();

SELECT set_config('kea.disable_audit', 'false', false);

-- Update the schema version number.
UPDATE schema_version
    SET version = '27', minor = '0';

-- Commit the script transaction.
COMMIT;

EOF
