// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F257

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_96_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x0eb4f2a1u, 0x0d283731u, 0, 6 }, // 0eb4f2a10d283731 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xda735b81u, 0x21b96d28u, 0, 143 }, // da735b8121b96d28 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xbd32bc52u, 0x9ac5f456u, 23, 109 }, // bd32bc529ac5f456 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xd59b7925u, 0xddcdc526u, 23, 6 }, // d59b7925ddcdc526 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x93cd4c9du, 0x6a39a3c6u, 23, 143 }, // 93cd4c9d6a39a3c6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x78e78984u, 0xd0683d17u, 63, 6 }, // 78e78984d0683d17 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x83dfd6aeu, 0x92245217u, 103, 109 }, // 83dfd6ae92245217 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xbd56f5fcu, 0xb6d93896u, 103, 6 }, // bd56f5fcb6d93896 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xd8e23adbu, 0x0e9ed020u, 103, 69 }, // d8e23adb0e9ed020 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x80d5fe24u, 0x750a8e0au, 160, 109 }, // 80d5fe24750a8e0a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x280c5aa2u, 0xbd6ce5b6u, 166, 109 }, // 280c5aa2bd6ce5b6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_F_1___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 4, 8, 6, 9, 2, 7, 7},
 { 3, 8, 2, 2, 7, 7, 7},
 { 3, 2, 4, 4, 5, 4, 4},
 { 2, 2, 4, 4, 4, 4, 4},
 { 4, 4, 5, 4, 4, 4, 4},
 { 4, 4, 4, 4, 4, 4, 5},
 { 4, 7, 4, 4, 4,10,10}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F257 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 96 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

