#include "taskchampion-cpp/lib.h"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 202002L
#include <ranges>
#endif

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BOX
#define CXXBRIDGE1_RUST_BOX
template <typename T>
class Box final {
public:
  using element_type = T;
  using const_pointer =
      typename std::add_pointer<typename std::add_const<T>::type>::type;
  using pointer = typename std::add_pointer<T>::type;

  Box() = delete;
  Box(Box &&) noexcept;
  ~Box() noexcept;

  explicit Box(const T &);
  explicit Box(T &&);

  Box &operator=(Box &&) & noexcept;

  const T *operator->() const noexcept;
  const T &operator*() const noexcept;
  T *operator->() noexcept;
  T &operator*() noexcept;

  template <typename... Fields>
  static Box in_place(Fields &&...);

  void swap(Box &) noexcept;

  static Box from_raw(T *) noexcept;

  T *into_raw() noexcept;

  /* Deprecated */ using value_type = element_type;

private:
  class uninit;
  class allocation;
  Box(uninit) noexcept;
  void drop() noexcept;

  friend void swap(Box &lhs, Box &rhs) noexcept { lhs.swap(rhs); }

  T *ptr;
};

template <typename T>
class Box<T>::uninit {};

template <typename T>
class Box<T>::allocation {
  static T *alloc() noexcept;
  static void dealloc(T *) noexcept;

public:
  allocation() noexcept : ptr(alloc()) {}
  ~allocation() noexcept {
    if (this->ptr) {
      dealloc(this->ptr);
    }
  }
  T *ptr;
};

template <typename T>
Box<T>::Box(Box &&other) noexcept : ptr(other.ptr) {
  other.ptr = nullptr;
}

template <typename T>
Box<T>::Box(const T &val) {
  allocation alloc;
  ::new (alloc.ptr) T(val);
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::Box(T &&val) {
  allocation alloc;
  ::new (alloc.ptr) T(std::move(val));
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::~Box() noexcept {
  if (this->ptr) {
    this->drop();
  }
}

template <typename T>
Box<T> &Box<T>::operator=(Box &&other) & noexcept {
  if (this->ptr) {
    this->drop();
  }
  this->ptr = other.ptr;
  other.ptr = nullptr;
  return *this;
}

template <typename T>
const T *Box<T>::operator->() const noexcept {
  return this->ptr;
}

template <typename T>
const T &Box<T>::operator*() const noexcept {
  return *this->ptr;
}

template <typename T>
T *Box<T>::operator->() noexcept {
  return this->ptr;
}

template <typename T>
T &Box<T>::operator*() noexcept {
  return *this->ptr;
}

template <typename T>
template <typename... Fields>
Box<T> Box<T>::in_place(Fields &&...fields) {
  allocation alloc;
  auto ptr = alloc.ptr;
  ::new (ptr) T{std::forward<Fields>(fields)...};
  alloc.ptr = nullptr;
  return from_raw(ptr);
}

template <typename T>
void Box<T>::swap(Box &rhs) noexcept {
  using std::swap;
  swap(this->ptr, rhs.ptr);
}

template <typename T>
Box<T> Box<T>::from_raw(T *raw) noexcept {
  Box box = uninit{};
  box.ptr = raw;
  return box;
}

template <typename T>
T *Box<T>::into_raw() noexcept {
  T *raw = this->ptr;
  this->ptr = nullptr;
  return raw;
}

template <typename T>
Box<T>::Box(uninit) noexcept {}
#endif // CXXBRIDGE1_RUST_BOX

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) & noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_RUST_OPAQUE
#define CXXBRIDGE1_RUST_OPAQUE
class Opaque {
public:
  Opaque() = delete;
  Opaque(const Opaque &) = delete;
  ~Opaque() = delete;
};
#endif // CXXBRIDGE1_RUST_OPAQUE

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
template <typename T, typename = void *>
struct operator_new {
  void *operator()(::std::size_t sz) { return ::operator new(sz); }
};

template <typename T>
struct operator_new<T, decltype(T::operator new(sizeof(T)))> {
  void *operator()(::std::size_t sz) { return T::operator new(sz); }
};
} // namespace detail

template <typename T>
union ManuallyDrop {
  T value;
  ManuallyDrop(T &&value) : value(::std::move(value)) {}
  ~ManuallyDrop() {}
};

template <typename T>
union MaybeUninit {
  T value;
  void *operator new(::std::size_t sz) { return detail::operator_new<T>{}(sz); }
  MaybeUninit() {}
  ~MaybeUninit() {}
};

namespace {
template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};
} // namespace
} // namespace cxxbridge1
} // namespace rust

namespace tc {
  struct Uuid;
  struct Operation;
  struct Replica;
  struct OptionTaskData;
  struct TaskData;
  struct PropValuePair;
  struct WorkingSet;
}

namespace tc {
#ifndef CXXBRIDGE1_STRUCT_tc$Uuid
#define CXXBRIDGE1_STRUCT_tc$Uuid
struct Uuid final {
  ::std::array<::std::uint8_t, 16> v;

  // Convert the given Uuid to a string.
  ::rust::String to_string() const noexcept;

  // Check whether this is the "nil" Uuid, used as a sentinel value.
  bool is_nil() const noexcept;

  bool operator==(Uuid const &) const noexcept;
  bool operator!=(Uuid const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_tc$Uuid

#ifndef CXXBRIDGE1_STRUCT_tc$Operation
#define CXXBRIDGE1_STRUCT_tc$Operation
struct Operation final : public ::rust::Opaque {
  // Check if this is a Create operation.
  bool is_create() const noexcept;

  // Check if this is a Update operation.
  bool is_update() const noexcept;

  // Check if this is a Delete operation.
  bool is_delete() const noexcept;

  // Check if this is an UndoPoint operation.
  bool is_undo_point() const noexcept;

  // Get the operation's uuid.
  //
  // Only valid for create, update, and delete operations.
  ::tc::Uuid get_uuid() const noexcept;

  // Get the `old_task` for this update operation.
  //
  // Only valid for delete operations.
  ::rust::Vec<::tc::PropValuePair> get_old_task() const noexcept;

  // Get the `property` for this update operation.
  //
  // Only valid for update operations.
  void get_property(::std::string &property_out) const noexcept;

  // Get the `value` for this update operation, returning false if the
  // `value` field is None.
  //
  // Only valid for update operations.
  bool get_value(::std::string &value_out) const noexcept;

  // Get the `old_value` for this update operation, returning false if the
  // `old_value` field is None.
  //
  // Only valid for update operations.
  bool get_old_value(::std::string &old_value_out) const noexcept;

  // Get the `timestamp` for this update operation.
  //
  // Only valid for update operations.
  ::std::int64_t get_timestamp() const noexcept;

  ~Operation() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_tc$Operation

#ifndef CXXBRIDGE1_STRUCT_tc$Replica
#define CXXBRIDGE1_STRUCT_tc$Replica
struct Replica final : public ::rust::Opaque {
  // Commit the given operations to the replica.
  void commit_operations(::rust::Vec<::tc::Operation> ops);

  // Commit the reverse of the given operations.
  bool commit_reversed_operations(::rust::Vec<::tc::Operation> ops);

  // Get `TaskData` values for all tasks in the replica.
  //
  // This contains `OptionTaskData` to allow C++ to `take` values out of the vector and use
  // them as `rust::Box<TaskData>`. Cxx does not support `Vec<Box<_>>`. Cxx also does not
  // handle `HashMap`, so the result is not a map from uuid to task. The returned Vec is
  // fully populated, so it is safe to call `take` on each value in the returned Vec once .
  ::rust::Vec<::tc::OptionTaskData> all_task_data();

  // Simiar to all_task_data, but returing only pending tasks (those in the working set).
  ::rust::Vec<::tc::OptionTaskData> pending_task_data();

  // Get the UUIDs of all tasks.
  ::rust::Vec<::tc::Uuid> all_task_uuids();

  // Expire old, deleted tasks.
  void expire_tasks();

  // Get an existing task by its UUID.
  ::tc::OptionTaskData get_task_data(::tc::Uuid uuid);

  // Get the operations for a task task by its UUID.
  ::rust::Vec<::tc::Operation> get_task_operations(::tc::Uuid uuid);

  // Return the operations back to and including the last undo point, or since the last sync if
  // no undo point is found.
  ::rust::Vec<::tc::Operation> get_undo_operations();

  // Get the number of local, un-sync'd operations, excluding undo operations.
  ::std::size_t num_local_operations();

  // Get the number of (un-synchronized) undo points in storage.
  ::std::size_t num_undo_points();

  // Rebuild the working set.
  void rebuild_working_set(bool renumber);

  // Get the working set for this replica.
  ::rust::Box<::tc::WorkingSet> working_set();

  // Sync with a server crated from `ServerConfig::Local`.
  void sync_to_local(::rust::String server_dir, bool avoid_snapshots);

  // Sync with a server created from `ServerConfig::Remote`.
  void sync_to_remote(::rust::String url, ::tc::Uuid client_id, ::std::string const &encryption_secret, bool avoid_snapshots);

  // Sync with a server created from `ServerConfig::Aws` using `AwsCredentials::Profile`.
  void sync_to_aws_with_profile(::rust::String region, ::rust::String bucket, ::rust::String profile_name, ::std::string const &encryption_secret, bool avoid_snapshots);

  // Sync with a server created from `ServerConfig::Aws` using `AwsCredentials::AccessKey`.
  void sync_to_aws_with_access_key(::rust::String region, ::rust::String bucket, ::rust::String access_key_id, ::rust::String secret_access_key, ::std::string const &encryption_secret, bool avoid_snapshots);

  // Sync with a server created from `ServerConfig::Aws` using `AwsCredentials::Default`.
  void sync_to_aws_with_default_creds(::rust::String region, ::rust::String bucket, ::std::string const &encryption_secret, bool avoid_snapshots);

  // Sync with a server created from `ServerConfig::Gcp`.
  //
  // An empty value for `credential_path` is converted to `Option::None`.
  void sync_to_gcp(::rust::String bucket, ::rust::String credential_path, ::std::string const &encryption_secret, bool avoid_snapshots);

  ~Replica() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_tc$Replica

#ifndef CXXBRIDGE1_STRUCT_tc$OptionTaskData
#define CXXBRIDGE1_STRUCT_tc$OptionTaskData
// Wrapper around `Option<Box<TaskData>>`, required since cxx does not support Option<T>.
//
// Note that if an OptionTaskData containing a task is dropped without calling `take`,
// it will leak the contained task. C++ code should be careful to always take.
struct OptionTaskData final {
  ::tc::TaskData *maybe_task;

  // Check if the value contains a task.
  bool is_some() const noexcept;

  // Check if the value does not contain a task.
  bool is_none() const noexcept;

  // Get the contained task, or panic if there is no task. The OptionTaskData
  // will be reset to contain None.
  ::rust::Box<::tc::TaskData> take() noexcept;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_tc$OptionTaskData

#ifndef CXXBRIDGE1_STRUCT_tc$TaskData
#define CXXBRIDGE1_STRUCT_tc$TaskData
struct TaskData final : public ::rust::Opaque {
  // Get the task's Uuid.
  ::tc::Uuid get_uuid() const noexcept;

  // Get a value on this task. If the property exists, returns true and updates
  // the output parameter. If not, returns false.
  bool get(::std::string const &property, ::std::string &value_out) const noexcept;

  // Check if the given property is set.
  bool has(::std::string const &property) const noexcept;

  // Enumerate all properties on this task, in arbitrary order.
  ::rust::Vec<::rust::String> properties() const noexcept;

  // Enumerate all properties and their values on this task, in arbitrary order, as a
  // vector.
  ::rust::Vec<::tc::PropValuePair> items() const noexcept;

  // Update the given property with the given value.
  void update(::std::string const &property, ::std::string const &value, ::rust::Vec<::tc::Operation> &ops) noexcept;

  // Like `update`, but removing the property by passing None for the value.
  void update_remove(::std::string const &property, ::rust::Vec<::tc::Operation> &ops) noexcept;

  // Delete the task. The name is `delete_task` because `delete` is a C++ keyword.
  void delete_task(::rust::Vec<::tc::Operation> &ops) noexcept;

  ~TaskData() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_tc$TaskData

#ifndef CXXBRIDGE1_STRUCT_tc$PropValuePair
#define CXXBRIDGE1_STRUCT_tc$PropValuePair
struct PropValuePair final {
  ::rust::String prop;
  ::rust::String value;

  bool operator==(PropValuePair const &) const noexcept;
  bool operator!=(PropValuePair const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_tc$PropValuePair

#ifndef CXXBRIDGE1_STRUCT_tc$WorkingSet
#define CXXBRIDGE1_STRUCT_tc$WorkingSet
struct WorkingSet final : public ::rust::Opaque {
  // Get the "length" of the working set: the total number of uuids in the set.
  ::std::size_t len() const noexcept;

  // Get the largest index in the working set, or zero if the set is empty.
  ::std::size_t largest_index() const noexcept;

  // True if the length is zero
  bool is_empty() const noexcept;

  // Get the uuid with the given index, if any exists. Returns the nil UUID if
  // there is no task at that index.
  ::tc::Uuid by_index(::std::size_t index) const noexcept;

  // Get the index for the given uuid, or zero if it is not in the working set.
  ::std::size_t by_uuid(::tc::Uuid uuid) const noexcept;

  // Get the entire working set, as a vector indexed by each task's id. For example, the
  // UUID for task 5 will be at `all_uuids()[5]`. All elements of the vector not corresponding
  // to a task contain the nil UUID.
  ::rust::Vec<::tc::Uuid> all_uuids() const noexcept;

  ~WorkingSet() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_tc$WorkingSet

extern "C" {
bool tc$cxxbridge1$Uuid$operator$eq(Uuid const &, Uuid const &) noexcept;

::tc::Uuid tc$cxxbridge1$uuid_v4() noexcept;

::tc::Uuid tc$cxxbridge1$uuid_from_string(::std::string const &uuid) noexcept;

void tc$cxxbridge1$Uuid$to_string(::tc::Uuid const &self, ::rust::String *return$) noexcept;

bool tc$cxxbridge1$Uuid$is_nil(::tc::Uuid const &self) noexcept;
::std::size_t tc$cxxbridge1$Operation$operator$sizeof() noexcept;
::std::size_t tc$cxxbridge1$Operation$operator$alignof() noexcept;

bool tc$cxxbridge1$Operation$is_create(::tc::Operation const &self) noexcept;

bool tc$cxxbridge1$Operation$is_update(::tc::Operation const &self) noexcept;

bool tc$cxxbridge1$Operation$is_delete(::tc::Operation const &self) noexcept;

bool tc$cxxbridge1$Operation$is_undo_point(::tc::Operation const &self) noexcept;

::tc::Uuid tc$cxxbridge1$Operation$get_uuid(::tc::Operation const &self) noexcept;

void tc$cxxbridge1$Operation$get_old_task(::tc::Operation const &self, ::rust::Vec<::tc::PropValuePair> *return$) noexcept;

void tc$cxxbridge1$Operation$get_property(::tc::Operation const &self, ::std::string &property_out) noexcept;

bool tc$cxxbridge1$Operation$get_value(::tc::Operation const &self, ::std::string &value_out) noexcept;

bool tc$cxxbridge1$Operation$get_old_value(::tc::Operation const &self, ::std::string &old_value_out) noexcept;

::std::int64_t tc$cxxbridge1$Operation$get_timestamp(::tc::Operation const &self) noexcept;

void tc$cxxbridge1$new_operations(::rust::Vec<::tc::Operation> *return$) noexcept;

void tc$cxxbridge1$add_undo_point(::rust::Vec<::tc::Operation> &ops) noexcept;
::std::size_t tc$cxxbridge1$Replica$operator$sizeof() noexcept;
::std::size_t tc$cxxbridge1$Replica$operator$alignof() noexcept;

::rust::repr::PtrLen tc$cxxbridge1$new_replica_in_memory(::rust::Box<::tc::Replica> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$new_replica_on_disk(::rust::String *taskdb_dir, bool create_if_missing, bool read_write, ::rust::Box<::tc::Replica> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$commit_operations(::tc::Replica &self, ::rust::Vec<::tc::Operation> *ops) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$commit_reversed_operations(::tc::Replica &self, ::rust::Vec<::tc::Operation> *ops, bool *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$all_task_data(::tc::Replica &self, ::rust::Vec<::tc::OptionTaskData> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$pending_task_data(::tc::Replica &self, ::rust::Vec<::tc::OptionTaskData> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$all_task_uuids(::tc::Replica &self, ::rust::Vec<::tc::Uuid> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$expire_tasks(::tc::Replica &self) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$get_task_data(::tc::Replica &self, ::tc::Uuid uuid, ::tc::OptionTaskData *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$get_task_operations(::tc::Replica &self, ::tc::Uuid uuid, ::rust::Vec<::tc::Operation> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$get_undo_operations(::tc::Replica &self, ::rust::Vec<::tc::Operation> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$num_local_operations(::tc::Replica &self, ::std::size_t *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$num_undo_points(::tc::Replica &self, ::std::size_t *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$rebuild_working_set(::tc::Replica &self, bool renumber) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$working_set(::tc::Replica &self, ::rust::Box<::tc::WorkingSet> *return$) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_local(::tc::Replica &self, ::rust::String *server_dir, bool avoid_snapshots) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_remote(::tc::Replica &self, ::rust::String *url, ::tc::Uuid client_id, ::std::string const &encryption_secret, bool avoid_snapshots) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_aws_with_profile(::tc::Replica &self, ::rust::String *region, ::rust::String *bucket, ::rust::String *profile_name, ::std::string const &encryption_secret, bool avoid_snapshots) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_aws_with_access_key(::tc::Replica &self, ::rust::String *region, ::rust::String *bucket, ::rust::String *access_key_id, ::rust::String *secret_access_key, ::std::string const &encryption_secret, bool avoid_snapshots) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_aws_with_default_creds(::tc::Replica &self, ::rust::String *region, ::rust::String *bucket, ::std::string const &encryption_secret, bool avoid_snapshots) noexcept;

::rust::repr::PtrLen tc$cxxbridge1$Replica$sync_to_gcp(::tc::Replica &self, ::rust::String *bucket, ::rust::String *credential_path, ::std::string const &encryption_secret, bool avoid_snapshots) noexcept;

bool tc$cxxbridge1$OptionTaskData$is_some(::tc::OptionTaskData const &self) noexcept;

bool tc$cxxbridge1$OptionTaskData$is_none(::tc::OptionTaskData const &self) noexcept;

::tc::TaskData *tc$cxxbridge1$OptionTaskData$take(::tc::OptionTaskData &self) noexcept;
::std::size_t tc$cxxbridge1$TaskData$operator$sizeof() noexcept;
::std::size_t tc$cxxbridge1$TaskData$operator$alignof() noexcept;

::tc::TaskData *tc$cxxbridge1$create_task(::tc::Uuid uuid, ::rust::Vec<::tc::Operation> &ops) noexcept;

::tc::Uuid tc$cxxbridge1$TaskData$get_uuid(::tc::TaskData const &self) noexcept;

bool tc$cxxbridge1$TaskData$get(::tc::TaskData const &self, ::std::string const &property, ::std::string &value_out) noexcept;

bool tc$cxxbridge1$TaskData$has(::tc::TaskData const &self, ::std::string const &property) noexcept;

void tc$cxxbridge1$TaskData$properties(::tc::TaskData const &self, ::rust::Vec<::rust::String> *return$) noexcept;

void tc$cxxbridge1$TaskData$items(::tc::TaskData const &self, ::rust::Vec<::tc::PropValuePair> *return$) noexcept;

void tc$cxxbridge1$TaskData$update(::tc::TaskData &self, ::std::string const &property, ::std::string const &value, ::rust::Vec<::tc::Operation> &ops) noexcept;

void tc$cxxbridge1$TaskData$update_remove(::tc::TaskData &self, ::std::string const &property, ::rust::Vec<::tc::Operation> &ops) noexcept;

void tc$cxxbridge1$TaskData$delete_task(::tc::TaskData &self, ::rust::Vec<::tc::Operation> &ops) noexcept;
bool tc$cxxbridge1$PropValuePair$operator$eq(PropValuePair const &, PropValuePair const &) noexcept;
::std::size_t tc$cxxbridge1$WorkingSet$operator$sizeof() noexcept;
::std::size_t tc$cxxbridge1$WorkingSet$operator$alignof() noexcept;

::std::size_t tc$cxxbridge1$WorkingSet$len(::tc::WorkingSet const &self) noexcept;

::std::size_t tc$cxxbridge1$WorkingSet$largest_index(::tc::WorkingSet const &self) noexcept;

bool tc$cxxbridge1$WorkingSet$is_empty(::tc::WorkingSet const &self) noexcept;

::tc::Uuid tc$cxxbridge1$WorkingSet$by_index(::tc::WorkingSet const &self, ::std::size_t index) noexcept;

::std::size_t tc$cxxbridge1$WorkingSet$by_uuid(::tc::WorkingSet const &self, ::tc::Uuid uuid) noexcept;

void tc$cxxbridge1$WorkingSet$all_uuids(::tc::WorkingSet const &self, ::rust::Vec<::tc::Uuid> *return$) noexcept;
} // extern "C"

bool Uuid::operator==(Uuid const &rhs) const noexcept {
  return tc$cxxbridge1$Uuid$operator$eq(*this, rhs);
}

bool Uuid::operator!=(Uuid const &rhs) const noexcept {
  return !(*this == rhs);
}

// Generate a new, random Uuid.
::tc::Uuid uuid_v4() noexcept {
  return tc$cxxbridge1$uuid_v4();
}

// Parse the given string as a Uuid, panicking if it is not valid.
::tc::Uuid uuid_from_string(::std::string const &uuid) noexcept {
  return tc$cxxbridge1$uuid_from_string(uuid);
}

::rust::String Uuid::to_string() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  tc$cxxbridge1$Uuid$to_string(*this, &return$.value);
  return ::std::move(return$.value);
}

bool Uuid::is_nil() const noexcept {
  return tc$cxxbridge1$Uuid$is_nil(*this);
}

::std::size_t Operation::layout::size() noexcept {
  return tc$cxxbridge1$Operation$operator$sizeof();
}

::std::size_t Operation::layout::align() noexcept {
  return tc$cxxbridge1$Operation$operator$alignof();
}

bool Operation::is_create() const noexcept {
  return tc$cxxbridge1$Operation$is_create(*this);
}

bool Operation::is_update() const noexcept {
  return tc$cxxbridge1$Operation$is_update(*this);
}

bool Operation::is_delete() const noexcept {
  return tc$cxxbridge1$Operation$is_delete(*this);
}

bool Operation::is_undo_point() const noexcept {
  return tc$cxxbridge1$Operation$is_undo_point(*this);
}

::tc::Uuid Operation::get_uuid() const noexcept {
  return tc$cxxbridge1$Operation$get_uuid(*this);
}

::rust::Vec<::tc::PropValuePair> Operation::get_old_task() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::tc::PropValuePair>> return$;
  tc$cxxbridge1$Operation$get_old_task(*this, &return$.value);
  return ::std::move(return$.value);
}

void Operation::get_property(::std::string &property_out) const noexcept {
  tc$cxxbridge1$Operation$get_property(*this, property_out);
}

bool Operation::get_value(::std::string &value_out) const noexcept {
  return tc$cxxbridge1$Operation$get_value(*this, value_out);
}

bool Operation::get_old_value(::std::string &old_value_out) const noexcept {
  return tc$cxxbridge1$Operation$get_old_value(*this, old_value_out);
}

::std::int64_t Operation::get_timestamp() const noexcept {
  return tc$cxxbridge1$Operation$get_timestamp(*this);
}

// Create a new vector of operations. It's also fine to construct a
// `rust::Vec<tc::Operation>` directly.
::rust::Vec<::tc::Operation> new_operations() noexcept {
  ::rust::MaybeUninit<::rust::Vec<::tc::Operation>> return$;
  tc$cxxbridge1$new_operations(&return$.value);
  return ::std::move(return$.value);
}

// Add an UndoPoint operation to the vector of operations. All other
// operation types should be added via `TaskData`.
void add_undo_point(::rust::Vec<::tc::Operation> &ops) noexcept {
  tc$cxxbridge1$add_undo_point(ops);
}

::std::size_t Replica::layout::size() noexcept {
  return tc$cxxbridge1$Replica$operator$sizeof();
}

::std::size_t Replica::layout::align() noexcept {
  return tc$cxxbridge1$Replica$operator$alignof();
}

// Create a new in-memory replica, such as for testing.
::rust::Box<::tc::Replica> new_replica_in_memory() {
  ::rust::MaybeUninit<::rust::Box<::tc::Replica>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$new_replica_in_memory(&return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

// Create a new replica stored on-disk.
::rust::Box<::tc::Replica> new_replica_on_disk(::rust::String taskdb_dir, bool create_if_missing, bool read_write) {
  ::rust::MaybeUninit<::rust::Box<::tc::Replica>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$new_replica_on_disk(&taskdb_dir, create_if_missing, read_write, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Replica::commit_operations(::rust::Vec<::tc::Operation> ops) {
  ::rust::ManuallyDrop<::rust::Vec<::tc::Operation>> ops$(::std::move(ops));
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$commit_operations(*this, &ops$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool Replica::commit_reversed_operations(::rust::Vec<::tc::Operation> ops) {
  ::rust::ManuallyDrop<::rust::Vec<::tc::Operation>> ops$(::std::move(ops));
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$commit_reversed_operations(*this, &ops$.value, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::tc::OptionTaskData> Replica::all_task_data() {
  ::rust::MaybeUninit<::rust::Vec<::tc::OptionTaskData>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$all_task_data(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::tc::OptionTaskData> Replica::pending_task_data() {
  ::rust::MaybeUninit<::rust::Vec<::tc::OptionTaskData>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$pending_task_data(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::tc::Uuid> Replica::all_task_uuids() {
  ::rust::MaybeUninit<::rust::Vec<::tc::Uuid>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$all_task_uuids(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Replica::expire_tasks() {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$expire_tasks(*this);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::tc::OptionTaskData Replica::get_task_data(::tc::Uuid uuid) {
  ::rust::MaybeUninit<::tc::OptionTaskData> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$get_task_data(*this, uuid, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::tc::Operation> Replica::get_task_operations(::tc::Uuid uuid) {
  ::rust::MaybeUninit<::rust::Vec<::tc::Operation>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$get_task_operations(*this, uuid, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::tc::Operation> Replica::get_undo_operations() {
  ::rust::MaybeUninit<::rust::Vec<::tc::Operation>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$get_undo_operations(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::size_t Replica::num_local_operations() {
  ::rust::MaybeUninit<::std::size_t> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$num_local_operations(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::size_t Replica::num_undo_points() {
  ::rust::MaybeUninit<::std::size_t> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$num_undo_points(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Replica::rebuild_working_set(bool renumber) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$rebuild_working_set(*this, renumber);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::rust::Box<::tc::WorkingSet> Replica::working_set() {
  ::rust::MaybeUninit<::rust::Box<::tc::WorkingSet>> return$;
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$working_set(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Replica::sync_to_local(::rust::String server_dir, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_local(*this, &server_dir, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Replica::sync_to_remote(::rust::String url, ::tc::Uuid client_id, ::std::string const &encryption_secret, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_remote(*this, &url, client_id, encryption_secret, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Replica::sync_to_aws_with_profile(::rust::String region, ::rust::String bucket, ::rust::String profile_name, ::std::string const &encryption_secret, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_aws_with_profile(*this, &region, &bucket, &profile_name, encryption_secret, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Replica::sync_to_aws_with_access_key(::rust::String region, ::rust::String bucket, ::rust::String access_key_id, ::rust::String secret_access_key, ::std::string const &encryption_secret, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_aws_with_access_key(*this, &region, &bucket, &access_key_id, &secret_access_key, encryption_secret, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Replica::sync_to_aws_with_default_creds(::rust::String region, ::rust::String bucket, ::std::string const &encryption_secret, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_aws_with_default_creds(*this, &region, &bucket, encryption_secret, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Replica::sync_to_gcp(::rust::String bucket, ::rust::String credential_path, ::std::string const &encryption_secret, bool avoid_snapshots) {
  ::rust::repr::PtrLen error$ = tc$cxxbridge1$Replica$sync_to_gcp(*this, &bucket, &credential_path, encryption_secret, avoid_snapshots);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool OptionTaskData::is_some() const noexcept {
  return tc$cxxbridge1$OptionTaskData$is_some(*this);
}

bool OptionTaskData::is_none() const noexcept {
  return tc$cxxbridge1$OptionTaskData$is_none(*this);
}

::rust::Box<::tc::TaskData> OptionTaskData::take() noexcept {
  return ::rust::Box<::tc::TaskData>::from_raw(tc$cxxbridge1$OptionTaskData$take(*this));
}

::std::size_t TaskData::layout::size() noexcept {
  return tc$cxxbridge1$TaskData$operator$sizeof();
}

::std::size_t TaskData::layout::align() noexcept {
  return tc$cxxbridge1$TaskData$operator$alignof();
}

// Create a new task with the given Uuid.
::rust::Box<::tc::TaskData> create_task(::tc::Uuid uuid, ::rust::Vec<::tc::Operation> &ops) noexcept {
  return ::rust::Box<::tc::TaskData>::from_raw(tc$cxxbridge1$create_task(uuid, ops));
}

::tc::Uuid TaskData::get_uuid() const noexcept {
  return tc$cxxbridge1$TaskData$get_uuid(*this);
}

bool TaskData::get(::std::string const &property, ::std::string &value_out) const noexcept {
  return tc$cxxbridge1$TaskData$get(*this, property, value_out);
}

bool TaskData::has(::std::string const &property) const noexcept {
  return tc$cxxbridge1$TaskData$has(*this, property);
}

::rust::Vec<::rust::String> TaskData::properties() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  tc$cxxbridge1$TaskData$properties(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Vec<::tc::PropValuePair> TaskData::items() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::tc::PropValuePair>> return$;
  tc$cxxbridge1$TaskData$items(*this, &return$.value);
  return ::std::move(return$.value);
}

void TaskData::update(::std::string const &property, ::std::string const &value, ::rust::Vec<::tc::Operation> &ops) noexcept {
  tc$cxxbridge1$TaskData$update(*this, property, value, ops);
}

void TaskData::update_remove(::std::string const &property, ::rust::Vec<::tc::Operation> &ops) noexcept {
  tc$cxxbridge1$TaskData$update_remove(*this, property, ops);
}

void TaskData::delete_task(::rust::Vec<::tc::Operation> &ops) noexcept {
  tc$cxxbridge1$TaskData$delete_task(*this, ops);
}

bool PropValuePair::operator==(PropValuePair const &rhs) const noexcept {
  return tc$cxxbridge1$PropValuePair$operator$eq(*this, rhs);
}

bool PropValuePair::operator!=(PropValuePair const &rhs) const noexcept {
  return !(*this == rhs);
}

::std::size_t WorkingSet::layout::size() noexcept {
  return tc$cxxbridge1$WorkingSet$operator$sizeof();
}

::std::size_t WorkingSet::layout::align() noexcept {
  return tc$cxxbridge1$WorkingSet$operator$alignof();
}

::std::size_t WorkingSet::len() const noexcept {
  return tc$cxxbridge1$WorkingSet$len(*this);
}

::std::size_t WorkingSet::largest_index() const noexcept {
  return tc$cxxbridge1$WorkingSet$largest_index(*this);
}

bool WorkingSet::is_empty() const noexcept {
  return tc$cxxbridge1$WorkingSet$is_empty(*this);
}

::tc::Uuid WorkingSet::by_index(::std::size_t index) const noexcept {
  return tc$cxxbridge1$WorkingSet$by_index(*this, index);
}

::std::size_t WorkingSet::by_uuid(::tc::Uuid uuid) const noexcept {
  return tc$cxxbridge1$WorkingSet$by_uuid(*this, uuid);
}

::rust::Vec<::tc::Uuid> WorkingSet::all_uuids() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::tc::Uuid>> return$;
  tc$cxxbridge1$WorkingSet$all_uuids(*this, &return$.value);
  return ::std::move(return$.value);
}
} // namespace tc

extern "C" {
void cxxbridge1$rust_vec$tc$PropValuePair$new(::rust::Vec<::tc::PropValuePair> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$PropValuePair$drop(::rust::Vec<::tc::PropValuePair> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$PropValuePair$len(::rust::Vec<::tc::PropValuePair> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$PropValuePair$capacity(::rust::Vec<::tc::PropValuePair> const *ptr) noexcept;
::tc::PropValuePair const *cxxbridge1$rust_vec$tc$PropValuePair$data(::rust::Vec<::tc::PropValuePair> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$PropValuePair$reserve_total(::rust::Vec<::tc::PropValuePair> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$tc$PropValuePair$set_len(::rust::Vec<::tc::PropValuePair> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$tc$PropValuePair$truncate(::rust::Vec<::tc::PropValuePair> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$tc$Operation$new(::rust::Vec<::tc::Operation> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$Operation$drop(::rust::Vec<::tc::Operation> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$Operation$len(::rust::Vec<::tc::Operation> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$Operation$capacity(::rust::Vec<::tc::Operation> const *ptr) noexcept;
::tc::Operation const *cxxbridge1$rust_vec$tc$Operation$data(::rust::Vec<::tc::Operation> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$Operation$reserve_total(::rust::Vec<::tc::Operation> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$tc$Operation$set_len(::rust::Vec<::tc::Operation> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$tc$Operation$truncate(::rust::Vec<::tc::Operation> *ptr, ::std::size_t len) noexcept;

::tc::Replica *cxxbridge1$box$tc$Replica$alloc() noexcept;
void cxxbridge1$box$tc$Replica$dealloc(::tc::Replica *) noexcept;
void cxxbridge1$box$tc$Replica$drop(::rust::Box<::tc::Replica> *ptr) noexcept;

void cxxbridge1$rust_vec$tc$OptionTaskData$new(::rust::Vec<::tc::OptionTaskData> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$OptionTaskData$drop(::rust::Vec<::tc::OptionTaskData> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$OptionTaskData$len(::rust::Vec<::tc::OptionTaskData> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$OptionTaskData$capacity(::rust::Vec<::tc::OptionTaskData> const *ptr) noexcept;
::tc::OptionTaskData const *cxxbridge1$rust_vec$tc$OptionTaskData$data(::rust::Vec<::tc::OptionTaskData> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$OptionTaskData$reserve_total(::rust::Vec<::tc::OptionTaskData> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$tc$OptionTaskData$set_len(::rust::Vec<::tc::OptionTaskData> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$tc$OptionTaskData$truncate(::rust::Vec<::tc::OptionTaskData> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$tc$Uuid$new(::rust::Vec<::tc::Uuid> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$Uuid$drop(::rust::Vec<::tc::Uuid> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$Uuid$len(::rust::Vec<::tc::Uuid> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$tc$Uuid$capacity(::rust::Vec<::tc::Uuid> const *ptr) noexcept;
::tc::Uuid const *cxxbridge1$rust_vec$tc$Uuid$data(::rust::Vec<::tc::Uuid> const *ptr) noexcept;
void cxxbridge1$rust_vec$tc$Uuid$reserve_total(::rust::Vec<::tc::Uuid> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$tc$Uuid$set_len(::rust::Vec<::tc::Uuid> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$tc$Uuid$truncate(::rust::Vec<::tc::Uuid> *ptr, ::std::size_t len) noexcept;

::tc::WorkingSet *cxxbridge1$box$tc$WorkingSet$alloc() noexcept;
void cxxbridge1$box$tc$WorkingSet$dealloc(::tc::WorkingSet *) noexcept;
void cxxbridge1$box$tc$WorkingSet$drop(::rust::Box<::tc::WorkingSet> *ptr) noexcept;

::tc::TaskData *cxxbridge1$box$tc$TaskData$alloc() noexcept;
void cxxbridge1$box$tc$TaskData$dealloc(::tc::TaskData *) noexcept;
void cxxbridge1$box$tc$TaskData$drop(::rust::Box<::tc::TaskData> *ptr) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::tc::PropValuePair>::Vec() noexcept {
  cxxbridge1$rust_vec$tc$PropValuePair$new(this);
}
template <>
void Vec<::tc::PropValuePair>::drop() noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$drop(this);
}
template <>
::std::size_t Vec<::tc::PropValuePair>::size() const noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$len(this);
}
template <>
::std::size_t Vec<::tc::PropValuePair>::capacity() const noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$capacity(this);
}
template <>
::tc::PropValuePair const *Vec<::tc::PropValuePair>::data() const noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$data(this);
}
template <>
void Vec<::tc::PropValuePair>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$reserve_total(this, new_cap);
}
template <>
void Vec<::tc::PropValuePair>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$tc$PropValuePair$set_len(this, len);
}
template <>
void Vec<::tc::PropValuePair>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$tc$PropValuePair$truncate(this, len);
}
template <>
Vec<::tc::Operation>::Vec() noexcept {
  cxxbridge1$rust_vec$tc$Operation$new(this);
}
template <>
void Vec<::tc::Operation>::drop() noexcept {
  return cxxbridge1$rust_vec$tc$Operation$drop(this);
}
template <>
::std::size_t Vec<::tc::Operation>::size() const noexcept {
  return cxxbridge1$rust_vec$tc$Operation$len(this);
}
template <>
::std::size_t Vec<::tc::Operation>::capacity() const noexcept {
  return cxxbridge1$rust_vec$tc$Operation$capacity(this);
}
template <>
::tc::Operation const *Vec<::tc::Operation>::data() const noexcept {
  return cxxbridge1$rust_vec$tc$Operation$data(this);
}
template <>
void Vec<::tc::Operation>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$tc$Operation$reserve_total(this, new_cap);
}
template <>
void Vec<::tc::Operation>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$tc$Operation$set_len(this, len);
}
template <>
void Vec<::tc::Operation>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$tc$Operation$truncate(this, len);
}
template <>
::tc::Replica *Box<::tc::Replica>::allocation::alloc() noexcept {
  return cxxbridge1$box$tc$Replica$alloc();
}
template <>
void Box<::tc::Replica>::allocation::dealloc(::tc::Replica *ptr) noexcept {
  cxxbridge1$box$tc$Replica$dealloc(ptr);
}
template <>
void Box<::tc::Replica>::drop() noexcept {
  cxxbridge1$box$tc$Replica$drop(this);
}
template <>
Vec<::tc::OptionTaskData>::Vec() noexcept {
  cxxbridge1$rust_vec$tc$OptionTaskData$new(this);
}
template <>
void Vec<::tc::OptionTaskData>::drop() noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$drop(this);
}
template <>
::std::size_t Vec<::tc::OptionTaskData>::size() const noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$len(this);
}
template <>
::std::size_t Vec<::tc::OptionTaskData>::capacity() const noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$capacity(this);
}
template <>
::tc::OptionTaskData const *Vec<::tc::OptionTaskData>::data() const noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$data(this);
}
template <>
void Vec<::tc::OptionTaskData>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$reserve_total(this, new_cap);
}
template <>
void Vec<::tc::OptionTaskData>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$tc$OptionTaskData$set_len(this, len);
}
template <>
void Vec<::tc::OptionTaskData>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$tc$OptionTaskData$truncate(this, len);
}
template <>
Vec<::tc::Uuid>::Vec() noexcept {
  cxxbridge1$rust_vec$tc$Uuid$new(this);
}
template <>
void Vec<::tc::Uuid>::drop() noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$drop(this);
}
template <>
::std::size_t Vec<::tc::Uuid>::size() const noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$len(this);
}
template <>
::std::size_t Vec<::tc::Uuid>::capacity() const noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$capacity(this);
}
template <>
::tc::Uuid const *Vec<::tc::Uuid>::data() const noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$data(this);
}
template <>
void Vec<::tc::Uuid>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$reserve_total(this, new_cap);
}
template <>
void Vec<::tc::Uuid>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$tc$Uuid$set_len(this, len);
}
template <>
void Vec<::tc::Uuid>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$tc$Uuid$truncate(this, len);
}
template <>
::tc::WorkingSet *Box<::tc::WorkingSet>::allocation::alloc() noexcept {
  return cxxbridge1$box$tc$WorkingSet$alloc();
}
template <>
void Box<::tc::WorkingSet>::allocation::dealloc(::tc::WorkingSet *ptr) noexcept {
  cxxbridge1$box$tc$WorkingSet$dealloc(ptr);
}
template <>
void Box<::tc::WorkingSet>::drop() noexcept {
  cxxbridge1$box$tc$WorkingSet$drop(this);
}
template <>
::tc::TaskData *Box<::tc::TaskData>::allocation::alloc() noexcept {
  return cxxbridge1$box$tc$TaskData$alloc();
}
template <>
void Box<::tc::TaskData>::allocation::dealloc(::tc::TaskData *ptr) noexcept {
  cxxbridge1$box$tc$TaskData$dealloc(ptr);
}
template <>
void Box<::tc::TaskData>::drop() noexcept {
  cxxbridge1$box$tc$TaskData$drop(this);
}
} // namespace cxxbridge1
} // namespace rust
