use crate::app::App;
use crate::error::NewsFlashGtkError;
use crate::gobject_models::GPluginID;
use crate::i18n::i18n;
use crate::util::GtkUtil;
use eyre::{Result, eyre};
use glib::{Object, Properties, SignalHandlerId, clone, prelude::*, subclass};
use gtk4::{Accessible, Buildable, CompositeTemplate, ConstraintTarget, Widget, subclass::prelude::*};
use libadwaita::{Bin, Toast, ToastOverlay, subclass::prelude::*};
use news_flash::error::NewsFlashError;
use news_flash::feed_api::FeedApiError;
use news_flash::models::{ApiSecret, LoginData, LoginGUI, OAuthData, PluginInfo};
use std::cell::RefCell;
use webkit6::{LoadEvent, WebView, prelude::*};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::WebLogin)]
    #[template(file = "data/resources/ui_templates/login/web.blp")]
    pub struct WebLogin {
        #[template_child]
        pub webview: TemplateChild<WebView>,

        #[template_child]
        pub toast_overlay: TemplateChild<ToastOverlay>,

        #[property(get, set, name = "plugin-id")]
        pub plugin_id: RefCell<GPluginID>,

        #[property(get, set, nullable, name = "custom-client-id")]
        pub custom_client_id: RefCell<Option<String>>,

        #[property(get, set, nullable, name = "custom-client-secret")]
        pub custom_client_secret: RefCell<Option<String>>,

        #[property(get, set, name = "catch-redirect-url")]
        pub catch_redirect_url: RefCell<String>,

        pub redirect_signal_id: RefCell<Option<SignalHandlerId>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for WebLogin {
        const NAME: &'static str = "WebLogin";
        type ParentType = Bin;
        type Type = super::WebLogin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for WebLogin {}

    impl WidgetImpl for WebLogin {}

    impl BinImpl for WebLogin {}
}

glib::wrapper! {
    pub struct WebLogin(ObjectSubclass<imp::WebLogin>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for WebLogin {
    fn default() -> Self {
        Object::new::<Self>()
    }
}

impl WebLogin {
    pub fn show_error(&self, error: NewsFlashError) {
        let imp = self.imp();

        let limit_reached = i18n("API Limit Reached");
        let login_failed = i18n("Failed to log in");
        let unknown_error = i18n("Unknown error");

        let error_text = if let NewsFlashError::API(FeedApiError::ApiLimit) = error {
            limit_reached
        } else if let NewsFlashError::NotLoggedIn = error {
            login_failed.clone()
        } else {
            unknown_error
        };

        let toast = Toast::new(&error_text);
        toast.set_button_label(Some(&i18n("details")));
        toast.set_action_name(Some("win.show-error-dialog"));
        imp.toast_overlay.add_toast(toast);

        App::default().set_newsflash_error(NewsFlashGtkError::NewsFlash {
            source: error,
            context: login_failed,
        });
    }

    pub fn set_service(&self, info: &PluginInfo) -> Result<()> {
        let LoginGUI::OAuth(web_login_desc) = &info.login_gui else {
            return Err(eyre!("No login OAuth GUI description"));
        };

        let plugin_id: GPluginID = info.id.clone().into();
        self.set_plugin_id(plugin_id);

        let Some(url) = (web_login_desc.login_website)(self.custom_api_secret().as_ref()) else {
            return Err(eyre!("No OAuth login URL"));
        };

        let Some(redirect_url) = web_login_desc.catch_redirect.clone() else {
            return Err(eyre!("No redirect URL"));
        };

        self.set_catch_redirect_url(redirect_url);

        let imp = self.imp();
        imp.webview.load_uri(url.as_str());

        let signal_id = imp.webview.connect_load_changed(clone!(
            #[weak(rename_to = this)]
            self,
            #[upgrade_or_panic]
            move |webview, event| {
                if event != LoadEvent::Started && event != LoadEvent::Redirected {
                    return;
                }

                let Some(uri) = webview.uri() else {
                    return;
                };

                let redirect_url = this.catch_redirect_url();

                if uri.len() > redirect_url.len() && uri[..redirect_url.len()] == redirect_url {
                    let oauth_data = OAuthData {
                        id: this.plugin_id().into(),
                        url: uri.as_str().to_owned(),
                        custom_api_secret: this.custom_api_secret(),
                    };
                    let oauth_data = LoginData::OAuth(oauth_data);
                    GtkUtil::disconnect_signal(this.imp().redirect_signal_id.take(), webview);
                    webview.stop_loading();

                    App::login(oauth_data);
                }
            }
        ));

        imp.redirect_signal_id.replace(Some(signal_id));
        Ok(())
    }

    pub fn reset(&self) {
        self.imp().webview.load_plain_text("");
    }

    fn custom_api_secret(&self) -> Option<ApiSecret> {
        if let (Some(client_id), Some(client_secret)) = (self.custom_client_id(), self.custom_client_secret()) {
            Some(ApiSecret {
                client_id,
                client_secret,
            })
        } else {
            None
        }
    }
}
