use crate::gobject_models::{GFeedListItem, GFeedListItemID, GSidebarSelection, GTag, GTagID, SidebarSelectionType};
use serde::{Deserialize, Serialize};

#[derive(Clone, Default, Debug, Serialize, Deserialize)]
pub enum SidebarSelection {
    #[default]
    All,
    Today,
    FeedList(GFeedListItemID, String, u32),
    Tag(GTagID, String, u32),
}

impl SidebarSelection {
    pub fn from_feed_list(item: GFeedListItem, pos: u32) -> Self {
        Self::FeedList(item.item_id(), item.label(), pos)
    }

    pub fn from_tag_list(tag: GTag, pos: u32) -> Self {
        Self::Tag(tag.tag_id(), tag.label(), pos)
    }
}

impl From<GSidebarSelection> for SidebarSelection {
    fn from(value: GSidebarSelection) -> Self {
        match value.selection_type() {
            SidebarSelectionType::All | SidebarSelectionType::None => Self::All,
            SidebarSelectionType::Today => Self::Today,
            SidebarSelectionType::FeedList => {
                Self::FeedList(value.feedlist_id(), value.label().unwrap_or_default(), value.position())
            }
            SidebarSelectionType::TagList => {
                Self::Tag(value.tag_id(), value.label().unwrap_or_default(), value.position())
            }
        }
    }
}

impl PartialEq for SidebarSelection {
    fn eq(&self, other: &SidebarSelection) -> bool {
        match self {
            SidebarSelection::All => matches!(other, SidebarSelection::All),
            SidebarSelection::Today => matches!(other, SidebarSelection::Today),
            SidebarSelection::FeedList(self_id, _title, _pos) => match other {
                SidebarSelection::FeedList(other_id, _title, _pos) => self_id == other_id,
                _ => false,
            },
            SidebarSelection::Tag(self_id, _title, _pos) => match other {
                SidebarSelection::Tag(other_id, _title, _pos) => self_id == other_id,
                _ => false,
            },
        }
    }
}
