use super::share_row::ShareRow;
use crate::app::App;
use crate::article_view::ArticleView;
use crate::content_page::ContentPage;
use crate::i18n::i18n;
use crate::main_window::MainWindow;
use crate::util::constants;
use glib::{Properties, prelude::*, subclass};
use gtk4::{CompositeTemplate, ListBox, Popover, Widget, prelude::*, subclass::prelude::*};
use percent_encoding::{NON_ALPHANUMERIC, utf8_percent_encode};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::SharePopover)]
    #[template(file = "data/resources/ui_templates/share/popover.blp")]
    pub struct SharePopover {
        #[template_child]
        pub list: TemplateChild<ListBox>,

        #[property(get, set, name = "pocket-enabled")]
        pub pocket_enabled: Cell<bool>,

        #[property(get, set, name = "instapaper-enabled")]
        pub instapaper_enabled: Cell<bool>,

        #[property(get, set, name = "mastodon-enabled")]
        pub mastodon_enabled: Cell<bool>,

        #[property(get, set, name = "reddit-enabled")]
        pub reddit_enabled: Cell<bool>,

        #[property(get, set, name = "telegram-enabled")]
        pub telegram_enabled: Cell<bool>,

        #[property(get, set, name = "custom-enabled")]
        pub custom_enabled: Cell<bool>,

        #[property(get, set, name = "custom-name", nullable)]
        pub custom_name: RefCell<Option<String>>,

        #[property(get, set, name = "custom-url", nullable)]
        pub custom_url: RefCell<Option<String>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SharePopover {
        const NAME: &'static str = "SharePopover";
        type ParentType = Popover;
        type Type = super::SharePopover;

        fn class_init(klass: &mut Self::Class) {
            ShareRow::ensure_type();

            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SharePopover {
        fn constructed(&self) {
            let obj = self.obj();
            let share_settings = App::default().settings().share();

            share_settings
                .bind_property("pocket-enabled", &*obj, "pocket-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("instapaper-enabled", &*obj, "instapaper-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("mastodon-enabled", &*obj, "mastodon-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("reddit-enabled", &*obj, "reddit-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("telegram-enabled", &*obj, "telegram-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("custom-enabled", &*obj, "custom-enabled")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("custom-name", &*obj, "custom-name")
                .bidirectional()
                .sync_create()
                .build();

            share_settings
                .bind_property("custom-url", &*obj, "custom-url")
                .bidirectional()
                .sync_create()
                .build();
        }
    }

    impl WidgetImpl for SharePopover {}

    impl PopoverImpl for SharePopover {}

    #[gtk4::template_callbacks]
    impl SharePopover {
        #[template_callback]
        fn on_share_url_template(&self, row: &ShareRow) {
            let Some(article) = ArticleView::instance().article() else {
                ContentPage::instance().simple_message(&i18n("No article selected"));
                return;
            };

            let Some(url) = article.url() else {
                ContentPage::instance().simple_message(&i18n("Article does not have URL"));
                return;
            };

            let title = article.title();
            let title = title.unwrap_or(constants::NO_TITLE.to_string());

            let Some(url_template) = row.url_template() else {
                ContentPage::instance().simple_message(&i18n("Article does not have an URL Template"));
                return;
            };

            let article_url_escaped = utf8_percent_encode(&url, NON_ALPHANUMERIC).to_string();
            let title_escaped = utf8_percent_encode(&title, NON_ALPHANUMERIC).to_string();
            let share_url = url_template.replace("${url}", &article_url_escaped);
            let share_url = share_url.replace("${title}", &title_escaped);

            App::open_url_in_default_browser(&share_url);
            self.obj().popdown();
        }

        #[template_callback]
        fn on_share_clipboard(&self, _row: &ShareRow) {
            let Some(article) = ArticleView::instance().article() else {
                ContentPage::instance().simple_message(&i18n("No article selected"));
                return;
            };

            let Some(article_url) = article.url() else {
                ContentPage::instance().simple_message(&i18n("Article does not have URL"));
                return;
            };

            MainWindow::instance().clipboard().set_text(article_url.as_str());
            self.obj().popdown();
        }
    }
}

glib::wrapper! {
    pub struct SharePopover(ObjectSubclass<imp::SharePopover>)
        @extends Widget, Popover;
}

impl Default for SharePopover {
    fn default() -> Self {
        glib::Object::new()
    }
}
