use crate::app::App;
use crate::error::NewsFlashGtkError;
use crate::gobject_models::GPluginID;
use crate::i18n::{i18n, i18n_f};
use eyre::{Result, eyre};
use gdk4::Texture;
use glib::{Object, Properties, prelude::*, subclass::*};
use gtk4::{CompositeTemplate, prelude::*, subclass::prelude::*};
use libadwaita::{EntryRow, Toast, ToastOverlay};
use news_flash::error::{FeedApiError, NewsFlashError};
use news_flash::models::{
    BasicAuth, DirectLogin, LoginData, LoginGUI, PasswordLogin as PasswordLoginData, PluginInfo, TokenLogin,
};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::PasswordLogin)]
    #[template(file = "data/resources/ui_templates/login/password.blp")]
    pub struct PasswordLogin {
        #[template_child]
        pub url_entry: TemplateChild<EntryRow>,
        #[template_child]
        pub user_entry: TemplateChild<EntryRow>,
        #[template_child]
        pub toast_overlay: TemplateChild<ToastOverlay>,

        #[property(get, set)]
        pub headline: RefCell<String>,

        #[property(get, set)]
        pub url: RefCell<String>,

        #[property(get, set)]
        pub user: RefCell<String>,

        #[property(get, set)]
        pub password: RefCell<String>,

        #[property(get, set)]
        pub token: RefCell<String>,

        #[property(get, set, name = "http-user")]
        pub http_user: RefCell<String>,

        #[property(get, set, name = "http-password")]
        pub http_password: RefCell<String>,

        #[property(get, set, nullable)]
        pub logo: RefCell<Option<Texture>>,

        #[property(get, set, name = "has-url")]
        pub has_url: Cell<bool>,

        #[property(get, set, name = "support-token")]
        pub support_token: Cell<bool>,

        #[property(get, set, name = "show-http-auth")]
        pub show_http_auth: Cell<bool>,

        #[property(get, set, name = "is-busy")]
        pub is_busy: Cell<bool>,

        #[property(get, set, name = "can-login")]
        pub can_login: Cell<bool>,

        #[property(get, set, name = "selected-login-type")]
        pub selected_login_type: Cell<u32>,

        #[property(get, set, name = "plugin-id")]
        pub plugin_id: RefCell<GPluginID>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for PasswordLogin {
        const NAME: &'static str = "PasswordLogin";
        type ParentType = gtk4::Box;
        type Type = super::PasswordLogin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for PasswordLogin {
        fn constructed(&self) {
            let obj = self.obj();

            obj.connect_url_notify(super::PasswordLogin::update_can_login);
            obj.connect_user_notify(super::PasswordLogin::update_can_login);
            obj.connect_password_notify(super::PasswordLogin::update_can_login);
            obj.connect_token_notify(super::PasswordLogin::update_can_login);
            obj.connect_is_busy_notify(super::PasswordLogin::update_can_login);
            obj.connect_selected_login_type_notify(super::PasswordLogin::update_can_login);
            obj.connect_has_url_notify(super::PasswordLogin::update_can_login);
            obj.connect_support_token_notify(super::PasswordLogin::update_can_login);
        }
    }

    impl WidgetImpl for PasswordLogin {}

    impl BoxImpl for PasswordLogin {}

    #[gtk4::template_callbacks]
    impl PasswordLogin {
        #[template_callback]
        fn busy_visible_child(&self, is_busy: bool) -> &'static str {
            if is_busy { "spinner" } else { "label" }
        }

        #[template_callback]
        fn is_user_password_visible(&self, selected_login_type: u32) -> bool {
            selected_login_type == 0
        }

        #[template_callback]
        fn is_token_visible(&self, selected_login_type: u32) -> bool {
            selected_login_type == 1
        }

        #[template_callback]
        fn on_login_clicked(&self) {
            let obj = self.obj();

            obj.set_is_busy(true);

            let plugin_id = self.plugin_id.borrow().as_ref().clone();
            let url: Option<String> = if self.has_url.get() {
                let mut url_text = obj.url();
                if !url_text.starts_with("http://") && !url_text.starts_with("https://") {
                    url_text.insert_str(0, "https://");
                }
                obj.set_url(url_text.as_str());

                Some(url_text)
            } else {
                None
            };

            let basic_auth = if obj.show_http_auth() && !obj.http_user().is_empty() {
                Some(BasicAuth {
                    user: obj.http_user(),
                    password: if obj.http_password().is_empty() {
                        None
                    } else {
                        Some(obj.http_password())
                    },
                })
            } else {
                None
            };

            let login_data = if obj.selected_login_type() == 0 {
                LoginData::Direct(DirectLogin::Password(PasswordLoginData {
                    id: plugin_id,
                    url,
                    user: obj.user(),
                    password: obj.password(),
                    basic_auth,
                }))
            } else {
                LoginData::Direct(DirectLogin::Token(TokenLogin {
                    id: plugin_id,
                    url,
                    token: obj.token(),
                    basic_auth,
                }))
            };

            App::login(login_data);
        }
    }
}

glib::wrapper! {
    pub struct PasswordLogin(ObjectSubclass<imp::PasswordLogin>)
        @extends gtk4::Widget, gtk4::Box;
}

impl Default for PasswordLogin {
    fn default() -> Self {
        Object::new::<Self>()
    }
}

impl PasswordLogin {
    pub fn set_service(&self, info: &PluginInfo) -> Result<()> {
        let LoginGUI::Direct(direct_login_desc) = &info.login_gui else {
            return Err(eyre!("No login GUI description"));
        };

        let plugin_id: GPluginID = info.id.clone().into();

        self.set_has_url(direct_login_desc.url);
        self.set_support_token(direct_login_desc.support_token_login);
        self.set_show_http_auth(direct_login_desc.http_auth);
        self.set_plugin_id(plugin_id);

        self.reset();

        // set Icon
        if let Some(icon) = info.icon.clone() {
            let bytes = glib::Bytes::from_owned(icon.into_data());
            let texture = gdk4::Texture::from_bytes(&bytes);
            self.set_logo(texture.ok());
        }

        // set headline
        self.set_headline(i18n_f("Log into {}", &[&info.name]));

        // set focus to first entry
        if self.has_url() {
            self.imp().url_entry.grab_focus();
        } else {
            self.imp().user_entry.grab_focus();
        }

        Ok(())
    }

    pub fn reset(&self) {
        self.set_is_busy(false);
        self.set_url("");
        self.set_user("");
        self.set_password("");
        self.set_token("");
        self.set_http_user("");
        self.set_http_password("");
        self.set_selected_login_type(0);
    }

    pub fn show_error(&self, error: NewsFlashError) {
        let imp = self.imp();

        self.set_is_busy(false);

        let message = match &error {
            NewsFlashError::API(api_err) => match &api_err {
                FeedApiError::Auth => {
                    self.set_show_http_auth(true);

                    let message = i18n("Unauthorized");
                    log::warn!("{message}");

                    let toast = Toast::new(&message);
                    imp.toast_overlay.add_toast(toast);
                    message
                }
                FeedApiError::Network(error) => {
                    let message = if let Some(status) = error.status() {
                        if status.as_u16() == 495 || status.as_u16() == 496 {
                            let message = i18n("No valid CA certificate available");
                            log::warn!("{message}");

                            let toast = Toast::new(&message);
                            toast.set_button_label(Some(&i18n("ignore future TLS errors")));
                            toast.set_action_name(Some("win.ignore-tls-errors"));
                            imp.toast_overlay.add_toast(toast);
                            Some(message)
                        } else {
                            None
                        }
                    } else {
                        None
                    };

                    if let Some(message) = message {
                        message
                    } else {
                        let message = i18n("Network Error");
                        log::warn!("{message}");

                        let toast = Toast::new(&message);
                        toast.set_button_label(Some(&i18n("details")));
                        toast.set_action_name(Some("win.show-error-dialog"));
                        imp.toast_overlay.add_toast(toast);
                        message
                    }
                }
                _ => {
                    let message = i18n("Could not log in");
                    log::warn!("{message}");

                    let toast = Toast::new(&message);
                    toast.set_button_label(Some(&i18n("details")));
                    toast.set_action_name(Some("win.show-error-dialog"));
                    imp.toast_overlay.add_toast(toast);
                    message
                }
            },
            _ => {
                let message = i18n("Unknown error");
                log::warn!("{message}");

                let toast = Toast::new(&message);
                toast.set_button_label(Some(&i18n("details")));
                toast.set_action_name(Some("win.show-error-dialog"));
                imp.toast_overlay.add_toast(toast);
                message
            }
        };

        App::default().set_newsflash_error(NewsFlashGtkError::NewsFlash {
            source: error,
            context: message,
        });
    }

    pub fn fill(&self, data: DirectLogin) {
        self.set_url("");
        self.set_user("");
        self.set_password("");
        self.set_token("");
        self.set_http_user("");
        self.set_http_password("");

        if let DirectLogin::Password(password_data) = &data {
            self.set_selected_login_type(0);
            self.set_url(password_data.url.as_deref().unwrap_or(""));
            self.set_user(password_data.user.as_str());
            self.set_password(password_data.password.as_str());
            self.fill_basic_auth(password_data.basic_auth.as_ref());
        } else if let DirectLogin::Token(token_data) = &data {
            self.set_selected_login_type(1);
            self.set_url(token_data.url.as_deref().unwrap_or(""));
            self.set_token(token_data.token.as_str());
            self.fill_basic_auth(token_data.basic_auth.as_ref());
        }
    }

    fn fill_basic_auth(&self, basic_auth: Option<&BasicAuth>) {
        if let Some(basic_auth) = basic_auth {
            self.set_http_user(basic_auth.user.as_str());
            self.set_show_http_auth(true);

            if let Some(password) = basic_auth.password.as_deref() {
                self.set_http_password(password);
            }
        }
    }

    fn update_can_login(&self) {
        let can_login = if self.is_busy() || (self.has_url() && self.url().is_empty()) {
            false
        } else if self.support_token() && self.selected_login_type() == 1 {
            !self.token().is_empty()
        } else {
            !self.user().is_empty() && !self.password().is_empty()
        };

        self.set_can_login(can_login);
    }
}
