use crate::util::CHANNEL_ERROR;
use futures::FutureExt;
use once_cell::sync::OnceCell;
use std::fmt::Debug;
use std::future::Future;
use std::sync::Arc;
use tokio::runtime::Runtime;
use tokio::sync::oneshot;

static RUNTIME: OnceCell<Arc<Runtime>> = OnceCell::new();

pub struct TokioRuntime;

impl TokioRuntime {
    pub fn instance() -> Arc<Runtime> {
        RUNTIME
            .get_or_init(|| {
                let rt = Runtime::new().expect("failed to create tokio runtime");
                Arc::new(rt)
            })
            .clone()
    }

    pub fn execute_with_callback<T, Fut, FTokio, FGLib>(async_fn: FTokio, callback: FGLib)
    where
        T: Send + 'static + Debug,
        Fut: Future<Output = T> + Send + 'static,
        FTokio: FnOnce() -> Fut + Send + 'static,
        FGLib: FnOnce(T) + 'static,
    {
        let (sender, receiver) = oneshot::channel::<T>();

        let f = async move {
            let res = async_fn().await;
            sender.send(res).expect(CHANNEL_ERROR);
        };

        let glib_future = receiver.map(|res| {
            if let Ok(res) = res {
                callback(res);
            }
        });

        Self::instance().spawn(f);
        glib::MainContext::default().spawn_local(glib_future);
    }

    pub fn execute_with_future_callback<T, Fut, Fut2, FTokio, FGLib>(async_fn: FTokio, callback: FGLib)
    where
        T: Send + 'static + Debug,
        Fut: Future<Output = T> + Send + 'static,
        Fut2: Future<Output = ()> + 'static,
        FTokio: FnOnce() -> Fut + Send + 'static,
        FGLib: FnOnce(T) -> Fut2 + 'static,
    {
        let (sender, receiver) = oneshot::channel::<T>();

        let f = async move {
            let res = async_fn().await;
            sender.send(res).expect("CHANNEL_ERROR");
        };

        Self::instance().spawn(f);
        glib::MainContext::default().spawn_local(async move {
            if let Ok(res) = receiver.await {
                callback(res).await;
            }
        });
    }
}
