use super::loader::{ArticleListLoadResult, ArticleListLoader, LoadType};
use crate::app::App;
use crate::article_list::{ArticleList, model::ArticleListModel};
use crate::content_page::{ArticleListColumn, ContentPage};
use crate::infrastructure::TokioRuntime;
use crate::settings::GOrderBy;
use crate::sidebar::SideBar;
use glib::{Object, clone, prelude::*, subclass::*};
use gtk4::{prelude::*, subclass::prelude::*};
use news_flash::error::NewsFlashError;
use news_flash::models::ArticleID;
use once_cell::sync::Lazy;
use std::cell::{Cell, RefCell};
use std::rc::Rc;

mod imp {
    use super::*;

    #[derive(Default)]
    pub struct ArticleListLoadQueue {
        pub next: Rc<RefCell<Option<ArticleListLoader>>>,
        pub is_loading: Rc<Cell<bool>>,
        pub custom_page_size: Cell<Option<i64>>,
        pub custom_article_to_load: RefCell<Option<ArticleID>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleListLoadQueue {
        const NAME: &'static str = "ArticleListLoadQueue";
        type Type = super::ArticleListLoadQueue;
    }

    impl ObjectImpl for ArticleListLoadQueue {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("new")
                        .param_types([ArticleListModel::static_type()])
                        .build(),
                    Signal::builder("update")
                        .param_types([ArticleListModel::static_type()])
                        .build(),
                    Signal::builder("extend")
                        .param_types([ArticleListModel::static_type()])
                        .build(),
                ]
            });
            SIGNALS.as_ref()
        }
    }

    impl ArticleListLoadQueue {
        pub(super) fn update_force(&self, force_new: bool) {
            // don't replace queued 'new' with an 'update'
            let is_new_queued = self
                .next
                .borrow()
                .as_ref()
                .map(|loader| loader.is_type(LoadType::New))
                .unwrap_or(false);
            if self.is_loading.get() && is_new_queued {
                log::debug!("skip update");
                return;
            }

            let page_size = self.custom_page_size.get();
            let custom_article = self.custom_article_to_load.borrow().clone();
            let loader = Self::build_loader(
                if force_new { LoadType::New } else { LoadType::Update },
                page_size,
                custom_article,
            );

            if self.is_loading.get() {
                if let Some(old) = self.next.borrow_mut().replace(loader) {
                    log::debug!("queue update - replaces old {:?}", old.get_type());
                } else {
                    log::debug!("queue update");
                }
            } else {
                self.execute_update(loader);
            }
        }

        pub(super) fn extend(&self) {
            // don't replace queued 'new' or 'update' with an 'extend'
            let is_new_or_update_queued = self
                .next
                .borrow()
                .as_ref()
                .map(|loader| loader.is_type(LoadType::Extend))
                .unwrap_or(false);
            if self.is_loading.get() && is_new_or_update_queued {
                log::debug!("skip extend");
                return;
            }

            let page_size = self.custom_page_size.get();
            let custom_article = self.custom_article_to_load.borrow().clone();
            let loader = Self::build_loader(LoadType::Extend, page_size, custom_article);

            if self.is_loading.get() {
                if let Some(old) = self.next.borrow_mut().replace(loader) {
                    log::debug!("queue extend - replaces old {:?}", old.get_type());
                } else {
                    log::debug!("queue extend");
                }
            } else {
                self.execute_extend(loader);
            }
        }

        fn build_loader(
            load_type: LoadType,
            page_size: Option<i64>,
            custom_article: Option<ArticleID>,
        ) -> ArticleListLoader {
            let article_list_mode = ArticleListColumn::instance().mode();
            log::debug!("build loader {load_type:?} for mode {article_list_mode:?}");

            let search_term = ArticleListColumn::instance().search_term();
            let search_term = if search_term.is_empty() {
                None
            } else {
                Some(search_term)
            };

            let last_article_date = if App::default().settings().article_list().order_by() == GOrderBy::Published {
                ArticleList::instance()
                    .get_last_row_model()
                    .map(|model| model.date().into())
            } else {
                let model = ArticleList::instance().get_last_row_model();
                model.map(|model| model.updated().as_ref().unwrap_or(*model.date().as_ref()))
            };

            ArticleListLoader::default()
                .load_type(load_type)
                .hide_furure_articles(App::default().settings().article_list().hide_future_articles())
                .search_term(search_term)
                .order(App::default().settings().article_list().order().into())
                .order_by(App::default().settings().article_list().order_by().into())
                .mode(article_list_mode)
                .page_size(page_size)
                .list_height(ArticleList::instance().height() as i64)
                .sidebar_selection(SideBar::instance().selection().into())
                .undo_action(ContentPage::instance().get_current_undo_action())
                .last_article_date(last_article_date)
                .loaded_article_ids(ArticleList::instance().loaded_article_ids())
                .restore_selected_id(custom_article)
        }

        fn execute_update(&self, loader: ArticleListLoader) {
            self.is_loading.set(true);
            log::debug!("execute update");

            TokioRuntime::execute_with_callback(
                move || async move {
                    let news_flash = App::news_flash();
                    let guard = news_flash.read().await;
                    let news_flash = guard.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                    loader.load(news_flash)
                },
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[upgrade_or_panic]
                    move |res: Result<ArticleListLoadResult, NewsFlashError>| {
                        if let Ok(load_result) = res {
                            let load_type = load_result.load_type();
                            let list_model = load_result.build_list_model();

                            if load_type == LoadType::New {
                                imp.emit_new(list_model);
                            } else {
                                imp.emit_update(list_model);
                            }
                        }

                        imp.queue_next();
                    }
                ),
            );
        }

        fn execute_extend(&self, loader: ArticleListLoader) {
            self.is_loading.set(true);
            log::debug!("execute extend");

            TokioRuntime::execute_with_callback(
                move || async move {
                    let news_flash = App::news_flash();
                    let guard = news_flash.read().await;
                    let news_flash = guard.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                    loader.load(news_flash)
                },
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[upgrade_or_panic]
                    move |res: Result<ArticleListLoadResult, NewsFlashError>| {
                        if let Ok(load_result) = res {
                            imp.emit_extend(load_result.build_list_model());
                        }

                        imp.queue_next();
                    }
                ),
            );
        }

        fn queue_next(&self) {
            if let Some(loader) = self.next.borrow_mut().take() {
                log::debug!("executing next item in queue");

                match loader.get_type() {
                    LoadType::New | LoadType::Update => self.execute_update(loader),
                    LoadType::Extend => self.execute_extend(loader),
                }
            }

            self.is_loading.set(false);
        }

        fn emit_new(&self, model: ArticleListModel) {
            log::debug!("emit new article list");
            self.obj().emit_by_name::<()>("new", &[&model]);
        }

        fn emit_update(&self, model: ArticleListModel) {
            log::debug!("emit update article list");
            self.obj().emit_by_name::<()>("update", &[&model]);
        }

        fn emit_extend(&self, model: ArticleListModel) {
            log::debug!("emit extend article list");
            self.obj().emit_by_name::<()>("extend", &[&model]);
        }
    }
}

glib::wrapper! {
    pub struct ArticleListLoadQueue(ObjectSubclass<imp::ArticleListLoadQueue>);
}

impl Default for ArticleListLoadQueue {
    fn default() -> Self {
        Object::new()
    }
}

impl ArticleListLoadQueue {
    pub fn update(&self) {
        self.imp().update_force(false);
    }

    pub fn update_forced(&self) {
        self.imp().update_force(true);
    }

    pub fn extend(&self) {
        self.imp().extend();
    }

    pub fn set_custom_page_size(&self, page_size: Option<i64>) {
        self.imp().custom_page_size.replace(page_size);
    }

    pub fn set_custom_article_to_load(&self, article_id: Option<ArticleID>) {
        self.imp().custom_article_to_load.replace(article_id);
    }
}
